/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#include "loggingclass.h"
#include <QCoreApplication>
#include <QDebug>
#include <QDir>
#include <QStorageInfo>
#include <QTextStream>
#include <math.h>

LoggingClass::LoggingClass(int logging_rate, QString directory, QString filename, QStringList columns)
	: logging_rate(logging_rate)
	, directory(directory)
	, filename(filename)
	, data_columns(columns)
{
	//create a list of entries filled with 0's equal to the amount of columns provided.
	for (int i = 0; i < data_columns.count(); i++)
	{
		dataMap.insert(i, "0");
	}

	logTimer = new QTimer();
	connect(logTimer, &QTimer::timeout, this, &LoggingClass::dataLogger);
}

LoggingClass::~LoggingClass()
{
	logfile->close();
}

bool LoggingClass::storage_available()
{
	bool storage_sufficient = true;
	QStorageInfo storage(directory);

//	qDebug() << storage.bytesAvailable();
//	if (storage.bytesAvailable() <= 7787608192)	//Test
	if (storage.bytesAvailable() <= 209715200)	//200MB
	{
		storage_sufficient = false;
	}

	emit signal_datalogging_storage_sufficient(storage_sufficient);
	return storage_sufficient;
}

bool LoggingClass::storage_full()
{
	bool storage_full = false;
	QStorageInfo storage(directory);

//	qDebug() << storage.bytesAvailable();
//	if (storage.bytesAvailable() <= 7787528192)	//Test
	if (storage.bytesAvailable() <= 52428800)	//50MB
	{
		storage_full = true;
	}

	return storage_full;
}

/* Delete the file with the smallest number at the end (following the current name convention)
 * Note: if no other files are available the current file is deleted. Writing continues from a blank slate, but the column names are lost.
 * Ideally logging should never reach this situation because there are warnings for the user that storage is running low. */
bool LoggingClass::free_up_storage()
{
	qDebug() << "Attempting to free up storage";
	QDir dir(directory);
	QStringList fileList = dir.entryList(QStringList() << filename + "_*.csv" ,QDir::Files);
//	qDebug() << fileList;

	QStorageInfo storage(directory);
//	qDebug() << "STORAGE LEFT:" << storage.bytesAvailable();

	//
	//TODO:
	//I'm afraid this might be quite slow if there are thousands of logs...
	//
	int lowest_file_identifier = __INT_MAX__;

	foreach(QString files, fileList)
	{
		QRegExp rx(filename + "_(\\d+).csv");
		if(rx.indexIn(files) != -1)
		{
			if (rx.cap(1).toInt() < lowest_file_identifier)
			{
				lowest_file_identifier = rx.cap(1).toInt();
			}
		}
	}

	QFile file_to_delete(directory + filename + "_" + QString::number(lowest_file_identifier) + ".csv");

	bool deleted = file_to_delete.remove();
	emit signal_datalogging_log_deleted(deleted, file_to_delete.fileName());

	if (deleted == true)
	{
		qDebug() << "Deleted old log: " << file_to_delete.fileName();
	}
	else
	{
		qDebug() << "Could not delete file" << file_to_delete.fileName();
	}

	return deleted;
}

/* generate a filename for the log with an appropriate number behind it */
QString LoggingClass::generate_filepath()
{
	QDir dir(directory);
	QStringList fileList = dir.entryList(QStringList() << "*.csv" ,QDir::Files);
	int highest_file_identifier = 0;

	foreach(QString files, fileList)
	{
		QRegExp rx(filename + "_(\\d+).csv");
		if(rx.indexIn(files) != -1)
		{
			if (rx.cap(1).toInt() > highest_file_identifier)
			{
				highest_file_identifier = rx.cap(1).toInt();
			}
		}
	}

	QString filepath = directory + "/" + filename + "_" + QString::number(highest_file_identifier + 1) + ".csv";
	qDebug() << filepath;
	return filepath;
}

/* Create a file and fill the first row with appropriate column names. */
bool LoggingClass::createFile()
{
	//Open the file to be written.
	logfile = new QFile;
	logfile->setFileName(generate_filepath());

	if(!writeColumns())
	{
		return false;
	}

	return true;
}

/* Returns the runtime of the datalogging activity in seconds.milliseconds (e.g. 1.042) */
QString LoggingClass::getRunTime()
{
	/* Stupid QString 'math' necessary to get around (assumed) floating point-related rounding issues. */
	qint64 elapsedTime = elapsedTimer.elapsed();
	QString seconds = QString::number(elapsedTime);
	seconds.chop(3);
	if (seconds == "")
	{
		seconds = "0";
	}

	float milliseconds = (elapsedTime % 1000) * 0.001;

	QString timeString = seconds + "." + QString::number(milliseconds, 'f', 3).remove(0,2);
//	qDebug() << timeString;
	return timeString;
}

/* Write the column names */
bool LoggingClass::writeColumns()
{
	if(logfile->open(QIODevice::ReadWrite |QIODevice::Append| QIODevice::Text))
	{
		QTextStream out(logfile);

		//First column is always Runtime
		out << "Runtime,";

		for (int i = 0; i < data_columns.count(); i++)
		{
			out << data_columns[i];

			if (i != data_columns.count() - 1)
			{
				out << ",";
			}
		}

		out << "\n";
		logfile->flush();
		logfile->close();
	}
	else
	{
		return false;
	}

	return true;
}

/* Write a row of data. */
bool LoggingClass::writeData()
{
	if(logfile->open(QIODevice::ReadWrite |QIODevice::Append| QIODevice::Text))
	{
		QTextStream out(logfile);
		//First data point is always Runtime
		out << getRunTime() + ",";

		for (int i = 0; i < data_columns.count(); i++)
		{
			out << dataMap.value(i);

			if (i != data_columns.count() - 1)
			{
				out << ",";
			}
		}

		out << "\n";
		logfile->flush();
		logfile->close();
	}
	else
	{
		return false;
	}

	return true;
}

/* Gets called every time the logTimer times out to write date to the log file. */
bool LoggingClass::dataLogger()
{
	if (!storage_available())	//Check for Low storage
	{
		if (storage_full())		//Check for Full storage
		{
			free_up_storage();
		}
	}

	if (firstRun)
	{
		if (!createFile())
		{
			enable_datalogging(false);
			return false;
		}

		//Exit true prematurely to skip WriteData() on the first run to avoid logging nonsense values full of 0's
		//It is assumed the data is updated before the logTimer times out again.
		firstRun = false;
		return true;
	}

	if (!writeData())
	{
		enable_datalogging(false);
		return false;
	}

	return true;
}

/* Starts the logTimer and handles status updates to the outside world. */
void LoggingClass::enable_datalogging(bool enable)
{
	if (enable == true)
	{
		firstRun = true;
		elapsedTimer.start();

		// Run the first datalogger activity immediately, instead of a 1-second delay.
		if (!dataLogger())
		{
			emit signal_datalogging_enable(false);
			return;
		}

		logTimer->start(logging_rate);
		emit signal_datalogging_enable(true);
	}
	else
	{
		logTimer->stop();
		emit signal_datalogging_enable(false);
	}
}

/* Fill dataMap entry by column index */
bool LoggingClass::setData(int column_index, QString Data)
{
	if (column_index < dataMap.count())
	{
		dataMap[column_index] = Data;
		return true;
	}
	qDebug() << "LoggingClass: could not set data" << data_columns.at(column_index);
	return false;
}

/* Fill dataMap entry by column name */
bool LoggingClass::setData(QString column_name, QString Data)
{
	if (data_columns.contains(column_name))
	{
		dataMap[data_columns.indexOf(column_name)] = Data;
		return true;
	}
	qDebug() << "LoggingClass: could not set data" << column_name;
	return false;
}
